#!/usr/bin/env python3


"""Applies Nivre and Nilsson's (2005) projectivization algorithm."""


import sys


def blocks(f):
    """Returns blocks separated by empty lines (as lists of lines)."""
    current_block = []
    for line in f:
        if not line.rstrip():
            yield current_block
            current_block = []
        else:
            current_block.append(line)
    if current_block:
        yield current_block


def read_conll(block):
    while block[0].startswith('#'):
        block = block[1:]
    ids = []
    forms = []
    lemmas = []
    uposs = []
    xposs = []
    featss = []
    heads = []
    deprels = []
    depss = []
    miscs = []
    for line in block:
        i, form, lemma, upos, xpos, feats, head, deprel, deps, misc = \
                line.split()
        ids.append(int(i))
        forms.append(form)
        lemmas.append(lemma)
        uposs.append(upos)
        xposs.append(xpos)
        featss.append(feats)
        heads.append(int(head))
        deprels.append(deprel)
        depss.append(deps)
        miscs.append(misc)
    assert ids == list(range(1, len(ids) + 1))
    return (ids, forms, lemmas, uposs, xposs, featss, heads, deprels, depss,
            miscs)


def write_conll(ids, forms, lemmas, uposs, xposs, featss, heads, deprels,
        depss, miscs):
    for fields in zip(ids, forms, lemmas, uposs, xposs, featss, heads, deprels,
            depss, miscs):
        line = '\t'.join(str(f) for f in fields)
        print(line)
    print()


def dominates(h, w, heads):
    """Returns whether h is an ancestor of w."""
    head_of_w = heads[w - 1]
    if head_of_w == 0:
        return False
    if head_of_w == h:
        return True
    return dominates(h, head_of_w, heads)


def is_projective(h, d, heads):
    """Returns whether the arc h -> d is projective."""
    raise NotImplementedError


def arc_length(arc):
    """For an arc (h, d), returns its length (the distance between h and d)."""
    h, d = arc
    return abs(h - d)


def find_smallest_nonprojective_arc(heads):
    """Find the smallest nonprojective arc.
    
    Returns the smallest nonprojective arc (h, d) in the dependency tree.
    Raises ValueError if the tree is projective.
    """
    nonprojective_arcs = []
    for d, h in enumerate(heads, start=1):
        if h != 0 and not is_projective(h, d, heads):
            nonprojective_arcs.append((h, d))
    return min(nonprojective_arcs, key=arc_length)


def projectivize(heads):
    """Projectivizes the dependency tree.

    Projectivizes the tree by repeatedly lifting the smallest nonprojective
    arc until the tree is projective.
    """
    raise NotImplementedError


if __name__ == '__main__':
    for block in blocks(sys.stdin):
        (ids, forms, lemmas, uposs, xposs, featss, heads, deprels, depss,
                miscs) = read_conll(block)
        projectivize(heads)
        write_conll(ids, forms, lemmas, uposs, xposs, featss, heads, deprels,
                depss, miscs)
